/**
 * SimConnector.java
 * @author dpeters
 * Created: Mar 28, 2006
 */
package test;

import java.util.logging.Logger;
import java.io.IOException;
import game.net.*;
import game.*;
import game.protocol.*;
/**
 * Initiate a connection to a simulator. This is needed because all 
 * SCORETeam methods are blocking and initGame doesn't get its reply until
 * <em>both</em> teams have connected.
 * 
 * @version $Revision: 316 $ $Date: 2008-01-22 23:16:24 -0330 (Tue, 22 Jan 2008) $
 *
 */
class SimConnector extends SCORETeam implements Runnable {

	public SimConnector(Student sim, Student team) {
		super();
		mPendingOp = ConOp.CONNECT;
		mState = ConState.NOTCONNECTED;
		mTeam = team;
		mSim = sim;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 *  
	 */
	public void run() {
		this.connect("localhost", mSim, mTeam);
		mPendingOp = ConOp.NONE;
		mState = ConState.HANDSHAKE;
		doOp();
		try {
			this.quit();
		} catch (Exception e) {
			log.warning(mClassName + ".run -- exception on quit: " + e);
		}
		mState = ConState.NOTCONNECTED;
	}
	
	/**
	 * Do the operations. This will loop until mPendingOp is DISCONNECT.
	 * If there's nothing to do it will wait.
	 */
	private synchronized void doOp() {
		Reply rep;
		try {
			while (mPendingOp != ConOp.DISCONNECT) {
				switch (mPendingOp) {
				case INIT:
					if (mState == ConState.HANDSHAKE) {
						rep = this.initGame(mSide);
						mSide = ((OnSideReply)rep).getSide();
						mState = ConState.PLAYING;
					} else {
						log.warning(mClassName + " invalid state for init: " + mState);
					}
					mPendingOp = ConOp.NONE;
					break;
					
				case END:
					if (mState == ConState.PLAYING) {
						rep = this.end();
						mState = ConState.HANDSHAKE;
					} else {
						log.warning(mClassName + " invalid state for end: " + mState);
					}
					mPendingOp = ConOp.NONE;
					break;
					
				case NONE: // wait for something else.
					wait();
					break;
					
				default:
					log.warning("Unexpected SimConnector operation: " + mPendingOp);
					break;
				}
				notifyAll();
			}
		} catch (Exception e) {
			log.warning(mClassName + ".doOp -- exception: " + e);
		}		
		
	}

	public synchronized boolean isReady() {
		return mState == ConState.HANDSHAKE || mState == ConState.PLAYING;
	}
	
	public synchronized void doInit(Side s) {
		mSide = s;
		mPendingOp = ConOp.INIT;
		notifyAll();
	}
	
	public synchronized void doEnd() {
		mPendingOp = ConOp.END;
		notifyAll();
	}
	
	public synchronized void doQuit() {
		mPendingOp = ConOp.DISCONNECT;
		notifyAll();
	}
	
	public synchronized boolean isGameRunning() {
		return mState == ConState.PLAYING;
	}
	
	public synchronized Side getSide() {
		return mSide;
	}
	
	public Reply sendAllRequestUntil(int tm) 
		throws IOException, ParseException
	{
		Reply rep;
		log.finest("Sending all request until " + tm + "\n");
		do {
			rep = this.getGameData();
		} while (rep instanceof AllReply && ((AllReply)rep).getTime() < tm);
		return rep;
	}

	private Student mSim;
	private Student mTeam;
	private Side mSide;
	private static Logger log = Logger.getLogger("Soccer.team");
	
	private enum ConState { NOTCONNECTED, HANDSHAKE, PLAYING };
	private ConState mState;
	private enum ConOp { CONNECT, INIT, END, DISCONNECT, NONE  };
	private ConOp mPendingOp;

	private static final String mClassName = "SimConnector";
}
