/**
 * SimTestBall.java
 * @author dennis
 * Created: Apr 6, 2006
 */
package test;

import java.util.logging.Logger;

import junit.framework.TestCase;
import java.io.IOException;

import game.protocol.*;
import game.*;
import simulator.model.*;

/**
 * @version $Revision: 316 $ $Date: 2008-01-22 23:16:24 -0330 (Tue, 22 Jan 2008) $
 */
public class SimTestBall extends SimTestCase {

	public SimTestBall(String name) {
		super(name);
	}

	/**
	 * Simple test to show how to write more test cases.
	 */
	public void testMyFirstTestCase() {
		final String tcName = "testMyFirstTestCase";
		log.entering(cName, tcName);
		log.info("Test: ball travelling straight for 4 sec.");
		try {
			// Put the ball 0.5 left of player (0,0).
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(3.0, 12.5)) ;
			
			// At 200 ms, kick parallel to the x axis with velocity 10.0 m/s
			TestMain.bSide.kick( 200, 0, 10.0, 0.0 ) ;
			
			// Track the ball for 4 seconds
			AllReply rep = (AllReply)TestMain.aSide.sendAllRequestUntil( 4200 );
			
			// Check the ball's position.
			// Since the acceleration at time t (in seconds) is -0.1 v(t).
			// we have d/dt v(t) = -0.1 v(t), with a solution of
			// v(t) = v0 * exp( -0.1 t) where v0 is the initial velocity.
			// Integrating v(t) from 0 to t gives the distance the ball goes in t seconds
			// if the initial velocity is v0 
			//      dist(t,v0) = 10 v0 (1 - exp( -0.1 t ))
			double dist = 10.0 * 10.0 * (1.0 - Math.exp( -0.1 * 4 ));
			assertEquals("Ball X position", dist+3.0, rep.getBallPosition().getX(), 1.00);
			assertEquals("Ball Y position", 12.5, rep.getBallPosition().getY(), 1.00);
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, "testMyFirstTestCase");
	}

	/**
	 * Test that place ball works correctly.
	 */
	public void test_place_ball() {
		final String tcName = "test_place_ball";
		log.entering(cName, tcName);
		log.info("Test: Place Ball.");
		try {
			TestMain.aSide.place_ball(0, SMVector.makeCartesian(3.0, 12.5));
			
			AllReply aRep = (AllReply)TestMain.aSide.getGameData();
			
			// Check positions of the ball - the fuzz is zero
			// since this part at least should be exact.
			CartesianVect ballPos = aRep.getBallPosition();
			assertEquals("Ball position X", 3.0, ballPos.getX());
			assertEquals("Ball position Y", 12.5, ballPos.getY());
			//			pass &= sufficiently_equal( ar.ball_x, 3.0, 0.0 ) ;
			//            pass &= sufficiently_equal( ar.ball_y, 12.5, 0.0 ) ;
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_ball_move_horizontal() {
		final String tcName = "test_ball_move_horizontal";
		log.entering(cName, tcName);
		log.info("Test: Horizontal Ball Movement.");
		try {
			// Parallel to x axis. ====================================           
			// Put the ball 0.5m to right of player (0,0).
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(3.0, 12.5) ) ;
			
			// At 200 ms, kick parallel to the x axis with velocity 10.0 m/s
			TestMain.bSide.kick( 1000, 0, 10.0, 0.0 ) ;
			
			// Track the ball for 4 seconds
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil(5000) ;
			
			// Check the ball's position.
			// Since the acceleration at time t (in seconds) is -0.1 v(t).
			// we have d/dt v(t) = -0.1 v(t), with a solution of
			// v(t) = v0 * exp( -0.1 t) where v0 is the initial velocity.
			// Integrating v(t) from 0 to t gives the distance the ball goes in t seconds
			// if the initial velocity is v0 
			//      dist(t,v0) = 10 v0 (1 - exp( -0.1 t ))
			double dist = 10.0 * 10.0 * (1.0 - Math.exp( -0.1 * 4 )) ;
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", dist+3.0, ballP.getX(), 1.00);
			assertEquals("Ball position Y", 12.5, ballP.getY(), 1.00);
			//           pass &= sufficiently_equal( ar.ball_x, dist+3.0, 1.00 ) ;
			//            pass &= sufficiently_equal( ar.ball_y, 12.5, 1.00 ) ;
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_ball_move_vertical() {
		final String tcName = "test_ball_move_vertical";
		log.entering(cName, tcName);
		log.info("Test: Vertical Ball Movement.");
		try {
			
			// // Parallel to y axis. ====================================           
			// Put the ball 0.5 m above player (0,0).
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(2.5, 13.0 )) ;
			
			// At 5 seconds, kick parallel to the y axis with velocity 2.0 m/s
			TestMain.bSide.kick( 6000, 0, 2.0, 0.0 ) ;
			
			// Track the ball for 4 seconds
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil(10000) ;
			
			// Check the ball's position.
			// Since the acceleration at time t (in seconds) is -0.1 v(t).
			// we have d/dt v(t) = -0.1 v(t), with a solution of
			// v(t) = v0 * exp( -0.1 t) where v0 is the initial velocity.
			// Integrating v(t) from 0 to t gives the distance the ball goes in t seconds
			// if the initial velocity is v0 
			//      dist(t,v0) = 10 v0 (1 - exp( -0.1 t ))
			double dist = 10.0 * 2.0 * (1.0 - Math.exp( -0.1 * 4 )) ;
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 2.5, ballP.getX(), 1.00);
			assertEquals("Ball position Y", dist + 13.0, ballP.getY(), 1.00);
			//            pass &= sufficiently_equal( ar.ball_x, 2.5, 1.00 ) ;
			//            pass &= sufficiently_equal( ar.ball_y, dist + 13.0, 1.00 ) ; 
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_kicking1() {
		final String tcName = "test_kicking1";
		log.entering(cName, tcName);
		log.info("Test: Ball Kicking 1.");
		try {
			// Make sure players outside of 100 cm have no effect on the ball. ==============           
            // Put the ball > 100 cm at 45 degrees from player.
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(3.22, 13.22 )) ;
            
            // At 200 ms, try to kick with velocity 10.0 m/s
			TestMain.bSide.kick( 200, 0, 10.0, 0.0 ) ;
            
            // Track the ball for 2 seconds
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil(2200) ;

            // Check the ball's position.  Should not have changed.
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 3.22, ballP.getX(), 1.00);
			assertEquals("Ball position Y", 13.22, ballP.getY(), 1.00);
//            pass &= sufficiently_equal( ar.ball_x, 3.22, 0.001 ) ;
//            pass &= sufficiently_equal( ar.ball_y, 13.22, 0.001 ) ;
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_kicking2() {
		final String tcName = "test_kicking2";
		log.entering(cName, tcName);
		log.info("Test: Ball Kicking 2.");
		try {
			// Kicking with some angle. =====================================================
			// Put the ball 45 degrees down and to the right of player (0,2)
			// but still within kicking range.
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(10.5, 17.0));

			// At 1000 ms, kick with a velocity of 10 m/s and an angle of +15 degrees.
			TestMain.bSide.kick(1000, 2, 10.0, +15.0 ) ;
            
            // Track the ball for 1 second.
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil(2000) ;
            
            // Check the ball's position.
            // Since the acceleration at time t (in seconds) is -0.1 v(t).
            // we have d/dt v(t) = -0.1 v(t), with a solution of
            // v(t) = v0 * exp( -0.1 t) where v0 is the initial velocity.
            // Integrating v(t) from 0 to t gives the distance the ball goes in t seconds
            // if the initial velocity is v0 
            //      dist(t,v0) = 10 v0 (1 - exp( -0.1 t ))
			//
			// This distance is at (-45 + 15) = -30 degrees) from the x-axis.

            double dist = 10.0 * 10.0 * (1.0 - Math.exp( -0.1 * 1 )) ;
			double xdist = dist * Math.cos(Math.PI/6);
			double ydist = dist * Math.sin(Math.PI/6);
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", xdist + 10.5, ballP.getX(), 1.00);
			assertEquals("Ball position Y", 17.0 - ydist, ballP.getY(), 1.00);
//            pass &= sufficiently_equal( ar.ball_x, xdist + 10.5, 0.25 ) ;
            // FIX DKP pass &= sufficiently_equal( ar.ball_x, xdist + 10.0, 0.25 ) ;
//            pass &= sufficiently_equal( ar.ball_y, 17.0 - ydist, 0.25 ) ;
            
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}

	private static final String cName = "SimTestBall";


}
