/**
 * SimTestBounce.java
 * @author dennis
 * Created: Apr 6, 2006
 */
package test;

import java.io.IOException;
import java.util.logging.Logger;

import game.protocol.*;
import game.*;
import simulator.model.*;
import junit.framework.TestCase;

/**
 * @version $Revision: 407 $ $Date: 2008-03-25 23:52:35 -0230 (Tue, 25 Mar 2008) $
 */
public class SimTestBounce extends SimTestCase {

	/**
	 * @param name
	 */
	public SimTestBounce(String name) {
		super(name);
		// TODO Auto-generated constructor stub
	}

	public void test_corner_bounce() {
		final String tcName = "test_corner_bounce";
		log.entering(cName, tcName);
		log.info("Test: Corner Bounce.");
		try {
			log.fine( "Ball goes at (2,1)" );
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(2, 1) ) ;
			
			log.fine( "Player (1,4) goes just right of the ball" ) ;
			TestMain.bSide.place_player( 0, 4, SMVector.makeCartesian(2.5, 1) ) ;
			
			log.fine( "At 1 s kick the ball with full force at 225 degrees.") ;
			TestMain.bSide.kick( 1000, 4, 10.0, 45.0 ) ;
			
			log.fine( "The ball should bounce off the lower wall at (1,0). Then off the left wall at position (0, .5).") ;
			log.fine( "It should end up at a 45 degree angle and arrive at location (2,2.5) after (if I'm right!) 0.96 seconds.") ;
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 1960 ) ;
			
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 2.0, ballP.getX(), 1.00 );
			assertEquals("Ball position Y", 2.5, ballP.getY(), 1.00);
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}

	public void test_ball_player_bounce() {
		final String tcName = "test_ball_player_bounce";
		log.entering(cName, tcName);
		log.info("Test: Ball Player Bounce.");
		try {
			// Bouncing the ball off of the player. =====================================================
			// Put the ball within kicking range of player (0,0).
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(3.04, 12.5) ) ;
			
			// At 1000 ms, try to kick with velocity 10.0 m/s
			// int time, int player, double velocity, double alpha
			double x, y, r;
			x = 9.6 - 3.04; y = 17.5 - 12.5; r = Math.sqrt(x*x + y*y);
			double alpha = Math.acos(x/r)*(180.0/Math.PI);
			TestMain.bSide.kick( 1000, 0, 5.0, alpha ) ;
			
			// Get the state of the ball
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil(5000) ;
			
			// Check to see if the angle of reflection of the ball 
			// off of the player is correct.
			log.fine("Ball should be at (6.227, 22.64)" );
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 6.23, ballP.getX(), 2.0 );
			assertEquals("Ball position Y", 22.64, ballP.getY(), 2.0);
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_player_hits_ball1() {
		final String tcName = "test_player_hits_ball1";
		log.entering(cName, tcName);
		log.info("Test: Ball Hits Player 1.");
		try {
			log.fine( "Put ball at 10 m left of player (1,0) so player (1,0) can bump into it.") ;
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(37.5, 12.5) ) ;
			
			log.fine("Accelerate player (1,0) for 4 seconds" ) ;
			TestMain.aSide.accelerate( 1000, 0, 1.0 ) ;
			TestMain.aSide.accelerate( 5000, 0, 0.0 ) ;
			
			log.fine("After 4 seconds of accceleration the player has moved 8 m and travels at 4 m/s.");
			log.fine("After 4.5 seconds, the player hits the ball. The ball's initial velocity should be 6 m/s") ;
			
			log.fine("We wait until 4 seconds after impact." ) ;
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 9500 ) ;
			
			// dist(t,v0) = 10 v0 (1 - exp( -0.1 t ))
			double dist = 10.0 * 6.0 * (1.0 - Math.exp( -0.1 * 4.0 )) ;
			
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 37.5 - dist, ballP.getX(), 1.00 );
			assertEquals("Ball position Y", 12.5, ballP.getY(), 1.00);
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_goalpost_bounce_NE() {
		final String tcName = "test_goalpost_bounce_NE";
		log.entering(cName, tcName);
		log.info("Test: NE Goalpost Bounce.");
		try {
			log.fine( "Ball goes at (45, 12.5)" );
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(45, 12.5) ) ;
			
			log.fine( "Player (1,4) goes just below of the ball" ) ;
			TestMain.bSide.place_player( 0, 4, SMVector.makeCartesian(45, 12) ) ;
			
			log.fine( "At 1 s kick the ball with full force at 45 degrees.") ;
			TestMain.bSide.kick( 1000, 4, 10.0, -45.0 ) ;
			
			log.fine( "The ball should bounce off the goal post and straight back at the player.") ;
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 2384 ) ;
			
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 47.447, ballP.getX(), 1.00 );
			assertEquals("Ball position Y", 14.947, ballP.getY(), 1.00);
			//TestMain.aSide.sendAllRequestUntil( 5000 ) ; // let me watch a little while
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}

	public void test_goalpost_bounce_SE() {
		final String tcName = "test_goalpost_bounce_SE";
		log.entering(cName, tcName);
		log.info("Test: SE Goalpost Bounce.");
		try {
			log.fine( "Ball goes at (45, 12.5)" );
			TestMain.bSide.place_ball( 0, SMVector.makeCartesian(45, 12.5) ) ;
			
			log.fine( "Player (1,4) goes just above of the ball" ) ;
			TestMain.bSide.place_player( 0, 4, SMVector.makeCartesian(45, 13) ) ;
			
			log.fine( "At 1 s kick the ball with full force at 45 degrees.") ;
			TestMain.bSide.kick( 1000, 4, 10.0, 45.0 ) ;
			
			log.fine( "The ball should bounce off the goal post and straight back at the player.") ;
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 2384 ) ;
			
			CartesianVect ballP = ar.getBallPosition();
			assertEquals("Ball position X", 47.447, ballP.getX(), 1.00 );
			assertEquals("Ball position Y", 10.053, ballP.getY(), 1.00);
			//TestMain.aSide.sendAllRequestUntil( 5000 ) ; // let me watch a little while
			
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_goalpost_bounce_NW() {
		final String tcName = "test_goalpost_bounce_NW";
		log.entering(cName, tcName);
		log.info("Test: NW Goalpost Bounce.");
		try {
			double bY = 17.5-5*Math.cos(Math.PI/6.)+0.15/2.*Math.sin(Math.PI/6);
			double bX = 5*Math.sin(Math.PI/6.)+0.15/2.*Math.cos(Math.PI/6);
			log.fine( "Ball goes at (" + bX + ", " + bY + ")" );
			TestMain.aSide.place_ball( 0, SMVector.makeCartesian(bX, bY) ) ;
			
			log.fine( "Player (0,4) goes just below of the ball" ) ;
			TestMain.bSide.place_player(0, 0, SMVector.makeCartesian(0, 0)); // Get goalie out of way.
			TestMain.aSide.place_player( 0, 4, SMVector.makeCartesian(bX, bY-0.5) ) ;
			
			log.fine( "At 1 s kick the ball with full force at 30 degrees.") ;
			TestMain.aSide.kick( 1000, 4, 10.0, 30.0 ) ;
			
			log.fine( "The ball should strike the goal post with a glancing blow and reflect.") ;
			// Impact w. post should occur after 513 ms at 9.5m/s ang. 120deg
			// Tangent to impact is at 75deg = (120-45), normal is 120+45=165deg.
			// Reflected vball = 6.718m/s ang.75 + 0.5*6.718 ang. -15 (=345)
			double spdBounce = 9.5*Math.cos(Math.PI/4);
			SMVector vRefPar = SMVector.makePolar(spdBounce, 75);
			SMVector vRefPerp = SMVector.makePolar(0.5*spdBounce, 345);
			SMVector vRef = SMVector.makeCartesian(vRefPar.getX()+vRefPerp.getX(), vRefPar.getY()+vRefPerp.getY());
			// 0.5s after bounce ball velocity should be in same direction, slowed by friction.
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 2013 ) ;
			
			PolarVect ballV = ar.getBallVelocity();
			// A small error in the actual point of bounce will make a big difference in the reflected
			// velocity, so we have very high tolerances here. 
			assertEquals("Ball velocity mag", vRef.getMagnitude()*Math.exp(-0.05), ballV.getMagnitude(), 1.5 );
			assertEquals("Ball velocity angle", vRef.getAngle(), ballV.getAngle(), 25.00);
			//TestMain.aSide.sendAllRequestUntil( 5000 ) ; // let me watch a little while
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}

	public void test_goalpost_bounce_SW() {
		final String tcName = "test_goalpost_bounce_SW";
		log.entering(cName, tcName);
		log.info("Test: SW Goalpost Bounce.");
		try {
			double bY = 7.5+5*Math.cos(Math.PI/6.)-0.15/2.*Math.sin(Math.PI/6);
			double bX = 5*Math.sin(Math.PI/6.)+0.15/2.*Math.cos(Math.PI/6);
			log.fine( "Ball goes at (" + bX + ", " + bY + ")" );
			TestMain.aSide.place_ball( 0, SMVector.makeCartesian(bX, bY) ) ;
			
			log.fine( "Player (0,4) goes just above of the ball" ) ;
			TestMain.bSide.place_player(0, 0, SMVector.makeCartesian(0, 0)); // Get goalie out of way.
			TestMain.aSide.place_player( 0, 4, SMVector.makeCartesian(bX, bY+0.5) ) ;
			
			log.fine( "At 1 s kick the ball with full force at -30 degrees.") ;
			TestMain.aSide.kick( 1000, 4, 10.0, -30.0 ) ;
			
			log.fine( "The ball should strike the goal post with a glancing blow and reflect.") ;
			// Impact w. post should occur after 513 ms at 9.5m/s ang. 120deg
			// Tangent to impact is at 285deg = (60+45+180), normal is 60-45=15deg.
			// Reflected vball = 6.718m/s ang.285 + 0.5*6.718 ang. 15
			double spdBounce = 9.5*Math.cos(Math.PI/4);
			SMVector vRefPar = SMVector.makePolar(spdBounce, 285);
			SMVector vRefPerp = SMVector.makePolar(0.5*spdBounce, 15);
			SMVector vRef = SMVector.makeCartesian(vRefPar.getX()+vRefPerp.getX(), vRefPar.getY()+vRefPerp.getY());
			// 0.5s after bounce ball velocity should be in same direction, slowed by friction.
			AllReply ar = (AllReply)TestMain.aSide.sendAllRequestUntil( 2013 ) ;
			
			PolarVect ballV = ar.getBallVelocity();
			// A small error in the actual point of bounce will make a big difference in the reflected
			// velocity, so we have very high tolerances here. 
			assertEquals("Ball velocity mag", vRef.getMagnitude()*Math.exp(-0.05), ballV.getMagnitude(), 1.5 );
			assertEquals("Ball velocity angle", vRef.getAngle(), ballV.getAngle(), 25.00);
			//TestMain.aSide.sendAllRequestUntil( 5000 ) ; // let me watch a little while
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}


	private static final String cName = "SimTestBounce";

}
