/**
 * SimTestPlayer.java
 * @author dennis
 * Created: March 28, 2006
 */
package test;

import java.io.IOException;

import game.protocol.*;
import game.*;
import simulator.model.*;

public class SimTestPlayer extends SimTestCase {

	public SimTestPlayer(String name) {
		super(name);
	}
	
	/**
	 * Test that place players works correctly.
	 */
	public void test_place_players() {
		final String tcName = "test_place_players";
		log.entering(cName, tcName);
		log.info("Test: Place Players.");
		try {
            // Place players for team WEST.
            TestMain.bSide.place_player( 0, 0, SMVector.makeCartesian(2.0, 10.0));
            TestMain.bSide.place_player( 0, 1, SMVector.makeCartesian(4.1234, 20.1234));
            TestMain.bSide.place_player( 0, 2, SMVector.makeCartesian(6.987654321, 6.987654321));
            TestMain.bSide.place_player( 0, 3, SMVector.makeCartesian(6.0, 15.0));
            TestMain.bSide.place_player( 0, 4, SMVector.makeCartesian(25.0, 12.5));

            // Place players for team EAST.
            TestMain.aSide.place_player( 0, 0, SMVector.makeCartesian(46.0, 15.0));
            TestMain.aSide.place_player( 0, 1, SMVector.makeCartesian(35.0, 20.0));
            TestMain.aSide.place_player( 0, 2, SMVector.makeCartesian(30.0, 3.0));
            TestMain.aSide.place_player( 0, 3, SMVector.makeCartesian(45.0, 2.0));
            TestMain.aSide.place_player( 0, 4, SMVector.makeCartesian(48.0, 23.0));
 
			AllReply rep = (AllReply)TestMain.aSide.getGameData();
			
			// Check positions of all the players.
			
            // team WEST
			checkPlayerPosition(Side.WEST, 0, 2.0, 10.0, 0.001, rep);
			checkPlayerPosition(Side.WEST, 1, 4.1234, 20.1234, 0.001, rep);
			checkPlayerPosition(Side.WEST, 2, 6.987654321, 6.987654321, 0.001, rep);
			checkPlayerPosition(Side.WEST, 3, 6.0, 15.0, 0.001, rep);
			checkPlayerPosition(Side.WEST, 4, 25.0, 12.5, 0.001, rep);

            // team EAST
			checkPlayerPosition(Side.EAST, 0, 46.0, 15.0, 0.001, rep);
			checkPlayerPosition(Side.EAST, 1, 35.0, 20.0, 0.001, rep);
			checkPlayerPosition(Side.EAST, 2, 30.0, 3.0, 0.001, rep);
			checkPlayerPosition(Side.EAST, 3, 45.0, 2.0, 0.001, rep);
			checkPlayerPosition(Side.EAST, 4, 48.0, 23.0, 0.001, rep);
            
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_player_move_straight() {
		final String tcName = "test_player_move_straight";
		log.entering(cName, tcName);
		log.info("Test: Straight Line Player Movement.");
		try {
			// Parallel to x axis. ====================================           
            // Give player (0,0) an acceleration of 0.7 m/s/s.
            ControlReply cr = (ControlReply)TestMain.bSide.accelerate( 0, 0, 0.7);
            
            // Track the player for 5 seconds
            AllReply ar = (AllReply)TestMain.bSide.sendAllRequestUntil(5000+cr.getTime());
            
            // Check the player's position.
			// For straight line motion and constant acceleration, the distance 
			// travelled and velocity gained in t seconds is given by:
			//
			//			d = v0*t + (1/2)*a*t^2
			// 
			// And since here, the player starts from rest:
			// 
			//			d = (1/2)*a*t^2

            double dist = 0.5 * 0.7 * (5*5);		// equal to 8.75
            checkPlayerPosition(Side.WEST, 0, dist+2.5, 12.5, 5*0.25, ar) ;

			// Make the player come to a stop again.
            TestMain.bSide.accelerate( 0, 0, -1.0);
			
			// Parallel to y axis. ========================================================           
			// Place player (1,3) near bottom of centre line.
			TestMain.aSide.place_player(10000, 3, SMVector.makeCartesian(12.5, 1.0));

			// Spin until player is pointing upwards, -45 degrees/s for 2 secs.
			TestMain.aSide.spin(10000, 3, -45);
            TestMain.aSide.spin(12000, 3, 0);

			// Track the player until the spinning is complete, and check orientation.
            // Track the player for 5 seconds
			ar = (AllReply)TestMain.aSide.sendAllRequestUntil(12000) ;
			PolarVect vel = ar.getPlayerVelocity(Side.EAST, 3);
			assertEquals("Player 3 direction", 90.0, vel.getAngle(), 10.0);
//			pass &= sufficiently_equal(ar.player_theta[1][3], 90, 10);

			// Accelerate in positive y direction at 1 m/s/s
            TestMain.aSide.accelerate(13000, 3, 1);
            
            // Track the player for 5 seconds
			ar = (AllReply)TestMain.aSide.sendAllRequestUntil(18000) ;
            
            // Check the player's position.
			dist = 0.5 * 1 * (5*5);	
			checkPlayerPosition(Side.EAST, 3, 12.5, dist + 1.0, 1.25, ar);

			// Make the player come to a stop again.
            TestMain.aSide.accelerate(0, 3, -5.0);

			// Diagonally ================================================================           
			// Place player (0,1) near bottom left-hand corner.
			TestMain.bSide.place_player(20000, 1, SMVector.makeCartesian(1.0, 1.0));

			// Spin until player is pointing at 60 degrees, 30 degrees/s for 2 secs.
			TestMain.bSide.spin(20000, 1, 30);
            TestMain.bSide.spin(22000, 1, 0);

			// Track the player until the spinning is complete, and check orientation.
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil(22000) ;
            vel = ar.getPlayerVelocity(Side.WEST, 1);
            assertEquals("Player 1 direction", 60.0, vel.getAngle(), 10.0);
//			pass &= sufficiently_equal(ar.player_theta[0][1], 60, 10);

			// Accelerate at 0.2 m/s/s in direction of orientation.
            TestMain.bSide.accelerate(23000, 1, 0.2);
            
            // Track the player for 5 seconds
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil(28000) ;
            
			// Check the player's position.  Find distance travelled as before, then
			// split into x and y components using the orientation.
			dist = 0.5 * 0.2 * (5*5);
			double xdist = dist * Math.cos(Math.PI/3);
			double ydist = dist * Math.sin(Math.PI/3);
			checkPlayerPosition(Side.WEST, 1, 1.0 + xdist, 1.0 + ydist, 1.25, ar);
//            pass &= sufficiently_equal(ar.player_x[0][1], 1.0 + xdist, 1.25 ) ;
//            pass &= sufficiently_equal(ar.player_y[0][1], 1.0 + ydist, 1.25 );

			// Make the player come to a stop again.
            TestMain.bSide.accelerate(0, 1, -5.0);
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_player_spin() {
		final String tcName = "test_player_spin";
		log.entering(cName, tcName);
		log.info("Test: Player Spinning.");
		try {
			// Positive spin. =======================================================           
            // Give player (0,4) a spin of 10 degrees/s at 50 ms.
            TestMain.bSide.spin(50, 4, 10.0);
           
            // Check that players are placed and oriented properly at start.
            AllReply ar = (AllReply)TestMain.bSide.sendAllRequestUntil(7050) ;
            
            // Check the player's position orientation with 10 degree tolerance.
			//
			//		theta = omega*t
			//
            PolarVect vel = ar.getPlayerVelocity(Side.WEST, 4);
            assertEquals("Player 4 direction", 10*7.0, vel.getAngle(), 10);

			// Negative spin (that is not a whole number). ==========================           
            // Give player (0,3) a spin of -12.2 degrees/s at 8000 ms.
            TestMain.bSide.spin(8000, 3, -12.2);
            
            // Track the player for 3 seconds
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil(11000) ;
            
            // Check orientation with 10 degree tolerance.
            vel = ar.getPlayerVelocity(Side.WEST, 3);
            assertEquals("Player 3 direction", 360+3*-12.2, vel.getAngle(), 10);
//			pass &= sufficiently_equal(ar.player_theta[0][3], 360+3*-12.2, 10);

			// Spin of 0. ===========================================================
			// Give player (0,2) a spin of 0 degrees/s at 12000 ms.
            TestMain.bSide.spin(12000, 2, 0.0);
            
            // Track the player for 2 seconds
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil(14000) ;
            
            // Check orientation with 10 degree tolerance.
            vel = ar.getPlayerVelocity(Side.WEST, 2);
            assertEquals("Player 3 direction", 0, vel.getAngle(), 10);
//			pass &= sufficiently_equal(ar.player_theta[0][2], 0, 10);
		  // do the test stuff.
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	public void test_circle() {
		final String tcName = "test_circle";
		log.entering(cName, tcName);
		log.info("Test: circle.");
		try {
            log.fine("Move players (0,1) and (0,2) out of the way");
            TestMain.bSide.place_player( 0, 1, SMVector.makeCartesian(0.0, 5.0) ) ;
            TestMain.bSide.place_player( 0, 2, SMVector.makeCartesian(0.0, 10.0) ) ;
            
            log.fine("Place player (0,0) at the origin.");
            TestMain.bSide.place_player( 0, 0, SMVector.makeCartesian(0.0, 0.0) ) ;
            
            log.fine("After 3 seconds, acclerate the player.  0.5 m/s/s");
            TestMain.bSide.accelerate( 3000, 0, 0.5 ) ;
            
            log.fine("After 8 seconds, shut off acceleration. Coast at 4 m/s");
            TestMain.bSide.accelerate( 11000, 0, 0 ) ;
            
            log.fine("At the same time, spin at 1 radian per second.");
            log.fine("This should make the player circle.");
            TestMain.bSide.spin( 11000, 0, 57.2958 ) ;
            
            log.fine("wait until velocity is 4 m/s");
            AllReply ar = (AllReply)TestMain.bSide.sendAllRequestUntil( 11000 ) ;
           
            // Where are we?
            CartesianVect p0 = ar.getPlayerPosition(Side.WEST, 0);
//            double x0 = ar.getPlayerPosition ;
//            double y0 = ar.player_y[0][0] ;
            
            // Let's check that the player is a the right place.
            assertEquals("Player X", 16.0, p0.getX(), 2.0);
            assertEquals("Player Y", 0.0, p0.getY(), 2.0);
//            pass &= sufficiently_equal( x0, 16.0, 2.0 ) ;
//            pass &= sufficiently_equal( y0, 0.0, 2.0 ) ;
            log.fine("Player reaches (16,0) after 8 seconds of acceleration.");
            log.fine("Should start circling.");
            
            // Wait another second
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil( 12000 ) ;
            
            // Where is it now?
            CartesianVect p1 = ar.getPlayerPosition(Side.WEST, 0);
            double x_del = p1.getX() - p0.getX();
            double y_del = p1.getY() - p0.getY();
            log.fine("After 1 s circling");
            assertEquals("Player X distance", 3.3659, x_del, 0.25);
            assertEquals("Player Y distance", 1.8388, y_del, 0.25);
//            pass &= sufficiently_equal( 3.3659, x_del, .25 ) ;
//            pass &= sufficiently_equal( 1.8388, y_del, .25 ) ;
           
            // Wait another 2 seconds
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil( 14000 ) ;
            p1 = ar.getPlayerPosition(Side.WEST, 0);
            x_del = p1.getX() - p0.getX();
            y_del = p1.getY() - p0.getY();
            log.fine("After 3 s circling");
            assertEquals("Player X distance", .56448, x_del, 0.75);
            assertEquals("Player Y distance", 7.96000, y_del, 0.75);
//            pass &= sufficiently_equal(  .56448, x_del, .75 ) ;
//            pass &= sufficiently_equal( 7.96000, y_del, .75 ) ;
            
            // Wait until the player has made a complete circle
            // This takes 2pi seconds
            ar = (AllReply)TestMain.bSide.sendAllRequestUntil( 11000+6283 ) ;
            p1 = ar.getPlayerPosition(Side.WEST, 0);
            x_del = p1.getX() - p0.getX();
            y_del = p1.getY() - p0.getY();
            log.fine("After 3 s circling");
            log.fine("After 2*pi s circling");
            assertEquals("Player X distance", 0, x_del, 1.58);
            assertEquals("Player Y distance", 0, y_del, 1.58);
//            pass &= sufficiently_equal( 0, x_del, 1.58 ) ;
//            pass &= sufficiently_equal( 0, y_del, 1.58 ) ;
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}

	public void test_spiral() {
		final String tcName = "test_spiral";
		log.entering(cName, tcName);
		log.info("Test: spiral.");
		Reply rep;
		try {
            log.fine("Move players (0,1) and (0,2) out of the way");
            rep = TestMain.bSide.place_player( 0, 1, SMVector.makeCartesian(0.0, 5.0) );
            assertEquals("Reply to place_player", Reply.ReplyType.CONTROL, rep.getType());
            rep = TestMain.bSide.place_player( 0, 2, SMVector.makeCartesian(0.0, 10.0) ) ;
            assertEquals("Reply to place_player", Reply.ReplyType.CONTROL, rep.getType());
           
            log.fine("Place player (0,0) at the origin.");
            rep = TestMain.bSide.place_player( 0, 0, SMVector.makeCartesian(0.0, 0.0) ) ;
            assertEquals("Reply to place_player", Reply.ReplyType.CONTROL, rep.getType());
            
            log.fine("After 1 second, acclerate the player.  0.5 m/s/s");
            rep = TestMain.bSide.accelerate( 2000, 0, 0.5 ) ;
            assertEquals("Reply to accelerate", Reply.ReplyType.CONTROL, rep.getType());
            
            log.fine("After 4 seconds, spin at +90 deg per second.");
            log.fine("This should make the player circle.");
            rep = TestMain.bSide.spin( 6000, 0, 90 ) ;
            assertEquals("Reply to spin", Reply.ReplyType.CONTROL, rep.getType());
            
            log.fine("wait for 4 seconds");
            rep = TestMain.bSide.sendAllRequestUntil( 6000 ) ;
            
            // Where are we?
            CartesianVect p0 = ((AllReply)rep).getPlayerPosition(TestMain.bSide.getSide(), 0);
            
            // Let's check that the player is a the right place.
            assertEquals("Player 0 X", 4.0, p0.getX(), 2.0);
            assertEquals("Player 0 Y", 0.0, p0.getY(), 2.0);
            log.fine("Player reaches (4,0) after 4 seconds of acceleration.");
            log.fine("Should start spiraling.");
            
            // Wait another second
            rep = TestMain.bSide.sendAllRequestUntil( 7000 ) ;
            
            // Where is it now?
            CartesianVect p1 = ((AllReply)rep).getPlayerPosition(TestMain.bSide.getSide(), 0);
            double x_del = p1.getX() - p0.getX();
            double y_del = p1.getY() - p0.getY();
            assertEquals("X movement after 1 s", 1.3889, x_del, .25);
            assertEquals("Y movement after 1 s", 1.4759, y_del, .25);
//            pass &= sufficiently_equal( 1.3889, x_del, .25 ) ;
//            pass &= sufficiently_equal( 1.4759, y_del, .25 ) ;
            log.fine("After 1 s spiraling");
           
            // Wait another 2 seconds
            rep = TestMain.bSide.sendAllRequestUntil( 9000 ) ;
            p1 = ((AllReply)rep).getPlayerPosition(TestMain.bSide.getSide(), 0);
            x_del = p1.getX() - p0.getX();
            y_del = p1.getY() - p0.getY();
            assertEquals("X movement after 3 s", -2.4308, x_del, .75);
            assertEquals("Y movement after 3 s", 1.0706, y_del, .75);
//            pass &= sufficiently_equal( -2.4308, x_del, .75 ) ;
//            pass &= sufficiently_equal(  1.0706, y_del, .75 ) ;
            log.fine("On track after 3 s spiraling");
            
            // Wait another 3 seconds.
            rep = TestMain.bSide.sendAllRequestUntil( 12000 ) ;
            p1 = ((AllReply)rep).getPlayerPosition(TestMain.bSide.getSide(), 0);
            x_del = p1.getX() - p0.getX();
            y_del = p1.getY() - p0.getY();
            assertEquals("X movement after 6 s", -0.40528, x_del, 1.5);
            assertEquals("Y movement after 6 s", 4.4563, y_del, 1.5);
//            pass &= sufficiently_equal( -0.40528, x_del, 1.5 ) ;
//            pass &= sufficiently_equal( 4.4563, y_del, 1.5 ) ;
            log.fine("After 6 s spiraling"); 
            
            log.fine("At this point the velocity should max out") ;
            log.fine("It will take 4 seconds to complete a circle") ;
            
            rep = TestMain.bSide.sendAllRequestUntil( 16000 );
            p1 = ((AllReply)rep).getPlayerPosition(TestMain.bSide.getSide(), 0);
            
            double x_final = p1.getX() - p0.getX();
            double y_final = p1.getY() - p0.getY();
            assertEquals("X movement after 10 s", x_del, x_final, 1.0);
            assertEquals("Y movement after 10 s", y_del, y_final, 1.0);
//            pass &= sufficiently_equal( x_final, x_del, 1.0 ) ;
//            pass &= sufficiently_equal( y_final, y_del, 1.0 ) ;
		} catch (IOException e) {
			fail("IOException:" + e);
		} catch (ParseException e) {
			fail("ParseException: " + e);
		}
		log.exiting(cName, tcName);		
	}
	
	private static final String cName = "SimTestPlayer";

}
